using System;
using System.IO;
using System.IO.Compression;
using System.Text;
using System.Diagnostics;
using System.Threading;
using System.Runtime.InteropServices;
using System.Collections.Generic;


namespace CSharpRecipes
{
	public class FileSystemIO
    {
        #region 12.1 Tworzenie, kopiowanie, przenoszenie lub usuwanie pliku
        public static void CreateCopyMoveDeleteFile()
		{
			FileStream fileStream = null;
            Directory.CreateDirectory(@"c:\usuwanie\test");
			if (!File.Exists(@"c:\usuwanie\test\test.txt"))
			{
				fileStream = File.Create(@"c:\usuwanie\test\test.txt");

				fileStream.Close();
			}
			
			FileInfo fileInfo = null;
			fileStream = null;
			if (!File.Exists (@"c:\usuwanie\test\test.txt"))
			{
				fileInfo = new FileInfo(@"c:\usuwanie\test\test.txt");
				fileStream = fileInfo.Create();

				fileStream.Close();
			}
			
			if (File.Exists(@"c:\usuwanie\test\test.txt"))
			{
				File.Copy(@"c:\usuwanie\test\test.txt ", 
					Directory.GetCurrentDirectory() + @"\test.txt", true);
                Thread.Sleep(2000);
			}

			fileInfo = new FileInfo(@"c:\usuwanie\test\test.txt");
			fileInfo.CopyTo(@"c:\test.txt", true);

			if (!File.Exists(Directory.GetCurrentDirectory() + @"\test.txt"))
			{
				File.Move(@"c:\usuwanie\test\test.txt ", 
					Directory.GetCurrentDirectory() + @"\test.txt");
			}

			fileInfo = new FileInfo(@"c:\usuwanie\test\test.txt");
			if (!File.Exists(@"c:\test.txt "))
			{
				fileInfo.MoveTo(@"c:\test.txt ");
			}

			if (File.Exists(Directory.GetCurrentDirectory() + @"\test.txt"))
			{
				File.Delete(Directory.GetCurrentDirectory() + @"\test.txt");
			}


			if(File.Exists(@"c:\usuwanie\test\test.txt"))
			{
				fileInfo = new FileInfo(@"c:\usuwanie\test\test.txt");
				fileInfo.Delete();
			}
            Directory.Delete(@"c:\usuwanie",true);
		}
		#endregion

        #region 12.2 Operacje na atrybutach plikw
        public static void ManipulateFileAttribs()
		{
			string path = Path.GetTempFileName();
			if (!File.Exists(path))
				File.Create(path);
				
			try
			{
				// Zastosowanie statycznych metod klasy File do operacji na znacznikach czasu i atrybutach
				DisplayFileAttr1(path);
				ModifyFileAttr1(path);


                // Zastosowanie metod egzemplarza klasy FileInfo do operacji na znacznikach czasu i atrybutach
				DisplayFileAttr2(path);
				ModifyFileAttr2(path);
				ViewModifyFileAttr(path);
			}
			catch(Exception e)
			{
				Console.WriteLine(e.ToString());
			}
		}

		public static void DisplayFileAttr1(string path)
		{
			Console.WriteLine(File.GetCreationTime(path).ToString());
			Console.WriteLine(File.GetLastAccessTime(path).ToString());
			Console.WriteLine(File.GetLastWriteTime(path).ToString());
		}

		public static void DisplayFileAttr2(string path)
		{
			FileInfo fileInfo = new FileInfo(path);

			Console.WriteLine(fileInfo.CreationTime.ToString());
			Console.WriteLine(fileInfo.LastAccessTime.ToString());
			Console.WriteLine(fileInfo.LastWriteTime.ToString());
		}

		public static void ModifyFileAttr1(string path)
		{
			File.SetCreationTime(path, DateTime.Parse(@"May 10, 2003"));
			File.SetLastAccessTime(path, DateTime.Parse(@"May 10, 2003"));
			File.SetLastWriteTime(path, DateTime.Parse(@"May 10, 2003"));
		}

		public static void ModifyFileAttr2(string path)
		{
			FileInfo fileInfo  = new FileInfo(path);

			DateTime dt = new DateTime(2001,2,8);
			fileInfo.CreationTime = dt;
			fileInfo.LastAccessTime = dt;
			fileInfo.LastWriteTime = dt; 
		}

		public static void ViewModifyFileAttr(string path)
		{
			if(File.Exists(path))
			{
				FileInfo fileInfo = new FileInfo(path);

				// Wywietlenie atrybutw pliku
				Console.WriteLine(fileInfo.Attributes.ToString());

				// Wywietlenie informacji, czy plik jest ukryty
                Console.WriteLine("Czy plik jest ukryty? = {0}",
                ((fileInfo.Attributes & FileAttributes.Hidden) == FileAttributes.Hidden));
                
				// Modyfikacja atrybutw pliku
				fileInfo.Attributes |= FileAttributes.Hidden;

                // Wywietlenie informacji, czy plik jest ukryty
                Console.WriteLine("Czy plik jest ukryty? = {0}",
                ((fileInfo.Attributes & FileAttributes.Hidden) == FileAttributes.Hidden));


			}
		}

		#endregion

		#region 12.3 Zmiana nazwy pliku
		public static void RenameFile()
		{
            //File.Delete(@"C:\temp\foo.txt");
			if(!Directory.Exists(@"c:\temp"))
				Directory.CreateDirectory(@"c:\temp");
            File.Delete(@"C:\temp\bar.dat");

			if (!File.Exists(@"c:\temp\foo.txt"))
				File.Create(@"c:\temp\foo.txt");

            RenameFile1(@"c:\temp\foo.txt",@"c:\temp\bar.dat");

			RenameFile1(@"c:\temp\bar.dat",@"c:\temp\foo.txt");

            FileInfo originalFile = new FileInfo(@"c:\temp\foo.txt");
            RenameFile2(originalFile,@"c:\temp\bar.dat");
		}

		public static void RenameFile1(string originalName, string newName)
		{
			File.Move(originalName, newName);
		}

		public static void RenameFile2(FileInfo originalFile, string newName)
		{
			originalFile.MoveTo(newName);
		}


		#endregion

        #region 12.4 Ustalanie, czy plik istnieje
        public static void DetermineFileExists()
		{
			File.Create(@"c:\temp\test.txt");
			if (File.Exists(@"c:\temp\test.txt"))
			{
				// Operacja na pliku
				Console.WriteLine("Znaleziony");
			}
		}
		#endregion

        #region 12.5 Wybr metody otwarcia pliku lub strumienia dla zapisu i (lub) odczytu
        public static void ChooseMethodOpenFileOrStream()
        {
            // utworzenie tymczasowego pliku, na ktrym bd wykonywane operacje
            string tempFile = Path.GetTempFileName();

            // FileStream
            using (FileStream fileStream = File.Open(tempFile, FileMode.Append))
            {
                    string text = "Witaj wiecie ";
                    byte[] bytes = Encoding.ASCII.GetBytes(text.ToCharArray());

                    // zapisywanie do pliku
                    fileStream.Write(bytes, 0, bytes.Length);

                    // StreamReader
                    using (StreamReader streamReader = new StreamReader(tempFile))
                    {

                        char[] chars = new char[64];
                        // odczytanie bloku znakw
                        streamReader.Read(chars, 0, 64);
                        string charsFound = new string(chars);
                        Console.WriteLine("Liczba znakw w strumieniu: {0}", charsFound);
                    }

                    // StreamWriter
                    StreamWriter streamWriter = null;
                    // otwarcie pliku do zapisu
                    streamWriter = new StreamWriter(tempFile, true);
                    // dopisanie tekstu
                    streamWriter.WriteLine(", oto StreamWriter!");

                    // BinaryWriter
                    long pos = 0;
                    int twentyFive = 25;

                    // uruchomenie strumienia BinaryWriter ze strumieniem bazowym StreamWriter,
                    // ktry jest ju otwarty
                    using (BinaryWriter binaryWriter = new BinaryWriter(streamWriter.BaseStream))
                    {
                        // przejcie na koniec pliku
                        pos = binaryWriter.Seek(0, SeekOrigin.End);
                        // zapisanie liczby 25
                        binaryWriter.Write(twentyFive);
                    }

                    // Nie mona wywoa metody Close na streamWriter, poniewa
                    // uycie instrukcji using na binaryWriter powoduje wywoanie
                    // metody binaryWriter.Dispose, ktra z kolei wywouje metod
                    // Dispose na wewntrznym odwoaniu do obiektu streamWriter przekazanym
                    // do konstruktora bineryWriter.

                    // BinaryReader 
                    using (StreamReader streamReader2 = new StreamReader(tempFile))
                    {
                        using (BinaryReader binaryReader = new BinaryReader(streamReader2.BaseStream))
                        {
                            //long pos = 0;
                            //int twentyFive = 25;

                            // przeniesienie wskanika na miejsce, w ktrym zapisano liczb
                            for (long i = 0; i < pos; i++)
                                binaryReader.ReadByte();
                            // odczytanie liczby (powinna mie warto 25)
                            int num = binaryReader.ReadInt32();
                            // czy to ta sama liczba...?
                            if (num == twentyFive)
                                Console.WriteLine("Powiodo si odczytanie liczby 25 ze strumienia");
                            else
                            Console.WriteLine("Odczytanie liczby 25 ze strumienia nie powiodo si");
                    }
                }
            }
        }

		#endregion

        #region 12.6 Losowy dostp do czci pliku
        public static void RandomAccessPartOfFile()
        {
            string theFirstLine = "To jest pierwszy wiersz.";
            CreateFile(theFirstLine, 1020304050);
            ModifyFile(1020304050);
            ReadFile(theFirstLine);


            long offsetValue = -20;
            using (FileStream fileStream = File.Open("data.txt", FileMode.Open, FileAccess.ReadWrite, FileShare.None))
            {
                using (StreamWriter streamWriter = new StreamWriter(fileStream))
                {
                    if ((offsetValue + streamWriter.BaseStream.Position) >= 0)
                    {
                        streamWriter.BaseStream.Seek(offsetValue, SeekOrigin.Current);
                    }
                    else
                    {
                        Console.WriteLine("Nie mona przej poza plik.");
                    }

                    if ((offsetValue + streamWriter.BaseStream.Length) >= 0)
                    {
                        streamWriter.BaseStream.Seek(offsetValue, SeekOrigin.End);
                    }
                    else
                    {
                        Console.WriteLine("Nie mona przej poza plik.");
                    }

                    if (offsetValue >= 0)
                    {
                        streamWriter.BaseStream.Seek(offsetValue, SeekOrigin.Begin);
                    }
                    else
                    {
                        Console.WriteLine("Nie mona przej poza plik.");
                    }
                }
            }
        }
        
        public static void CreateFile(string theFirstLine, int theSecondLine)
        {
            using (FileStream fileStream = new FileStream("data.txt",
                   FileMode.Create,
                   FileAccess.ReadWrite,
                   FileShare.None))
            {
                using (StreamWriter streamWriter = new StreamWriter(fileStream))
                {
                    streamWriter.WriteLine(theFirstLine);
                    streamWriter.WriteLine(theSecondLine);
                    streamWriter.Flush();
                }
            }
        }

        public static void ModifyFile(int theSecondLine)
        {
            // otwarcie pliku do odczytu/zapisu
            using (FileStream fileStream = File.Open("data.txt", FileMode.Open, FileAccess.ReadWrite, FileShare.None))
            {
                using (StreamWriter streamWriter = new StreamWriter(fileStream))
                {
                    // zapamitanie dugoci pierwszego wiersza
                    int offset = streamWriter.NewLine.Length;
                    // zapamitanie dugoci drugiego wiersza
                    offset += (theSecondLine.ToString().Length);
                    // obliczenie wartoci przeciwnej
                    offset = offset - (2 * offset);
                    // przeniesienie wskanika za pierwszy wiersz
                    streamWriter.BaseStream.Seek(offset, SeekOrigin.End);

                    StringBuilder sb = new StringBuilder("Wiersz dodany po przeniesieniu wskanika ");
                    sb.AppendFormat("{0} znakw od koca pliku.", offset);

                    streamWriter.WriteLine(sb.ToString());
                    streamWriter.Flush();

                    streamWriter.BaseStream.Seek(0, SeekOrigin.End);
                    streamWriter.WriteLine("Ostatni wiersz, dodany po przeniesieniu wskanika na koniec pliku.");
                }
            }
        }
        
        public static void ReadFile(string theFirstLine)
        {
            using (StreamReader streamReader = new StreamReader("data.txt"))
            {
                streamReader.BaseStream.Seek(theFirstLine.Length + Environment.NewLine.Length, SeekOrigin.Begin);

                Console.WriteLine(streamReader.ReadToEnd());
            }
        }

        public void MoveInFile(int offsetValue)
        {
            using (FileStream fileStream = File.Open("data.txt", FileMode.Open, FileAccess.ReadWrite, FileShare.None))
            {
                using (StreamWriter streamWriter = new StreamWriter(fileStream))
                {
                    // przesunicie wzgldem pocztku pliku
                    streamWriter.BaseStream.Seek(offsetValue, SeekOrigin.Begin);

                    // przesunicie wzgldem koca pliku
                    streamWriter.BaseStream.Seek(offsetValue, SeekOrigin.End);

                    // przesunicie wzgldem biecej lokalizacji wskanika w pliku
                    streamWriter.BaseStream.Seek(offsetValue, SeekOrigin.Current);
                }
            }
        }

		#endregion

        #region 12.7 Generowanie znaku EOL niezalenego od platformy
        public static void PlatformIndependentEOL()
		{
            // Naley pamita, by uywa Environment.NewLine w kadym formatowanym 
            // bloku tekstu, ktry powinien zawiera symbole nowego wiersza zgodne z platform
            string line;
            line = String.Format("PierwszyWiersz {0} DrugiWiersz {0} TrzeciWiersz {0}", Environment.NewLine);

            // wykorzystanie pliku tymczasowego
            string file = Path.GetTempFileName();
            using (FileStream stream = File.Create(file))
            {
                byte[] bytes = Encoding.Unicode.GetBytes(line);
                stream.Write(bytes, 0, bytes.Length);
            }

            // usunicie pliku (w tym wierszu mona ustawi punkt przerwania i sprawdzi
            // zawarto utworzonego pliku)
            File.Delete(file);


            // Zdefiniowanie obiektu zapisujcego tekst i wskazanie mu odpowiedniego symbolu nowego wiersza
            // wykorzystanie pliku tymczasowego
            file = Path.GetTempFileName();
            line = "Wiersz z podwjnym odstpem";
            using (StreamWriter streamWriter = new StreamWriter(file))
            {
                // przypisanie symbolowi nowego wiersza dwch znakw nowego wiersza
                streamWriter.NewLine = Environment.NewLine + Environment.NewLine;
                // metoda WriteLine strumienia bdzie automatycznie uywa wskazanej sekwencji nowego wiersza
                // (czyli podwjnego symbolu nowego wiersza)
                streamWriter.WriteLine(line);
                streamWriter.WriteLine(line);
                streamWriter.WriteLine(line);
            }
            // usunicie pliku (w tym wierszu mona ustawi punkt przerwania i sprawdzi
            // zawarto utworzonego pliku)
            File.Delete(file);

            // Uycie standardowych metod WriteLine, ktre domylnie korzystaj
            // z sekwencji Environment.NewLine
            line = "Wiersz domylny";
            Console.WriteLine(line);

		}
		#endregion

        #region 12.8 Tworzenie, zapisywanie do i odczytywanie z pliku
        public static void CreateWriteReadFile()
		{
			string tempFile = Path.GetTempFileName();
			FileStream fileStream = null;
			try
			{
				fileStream = CreateLogFile(tempFile);
				WriteToLog(fileStream,"Pierwsze zdarzenie dziennika!");
                WriteToLog(fileStream, "Drugie zdarzenie dziennika!");
                WriteToLog(fileStream, "Trzecie zdarzenie dziennika!");
				string logContents = ReadAllLog(fileStream);
				Console.WriteLine("Wynik jednoczesnego odczytania caej zawartoci pliku dziennika:\r\n{0}",logContents);
				ReadLogPeeking(fileStream);
				ReadLogByLines(fileStream);
				ReadAllLogAsBytes(fileStream);
				ReadAllBufferedLog(fileStream);
				fileStream.Close();
			}
			catch(Exception e)
			{
				Console.WriteLine(e.ToString());
			}


			StreamReader streamReader = new StreamReader(tempFile);
			while (streamReader.Peek() != -1)
			{
				Console.WriteLine(streamReader.ReadLine());
			}
			streamReader.Close();


			streamReader = new StreamReader(tempFile);
			char [] chars = new char[5];
			while ((streamReader.Read(chars, 0, chars.Length)) != 0)
			{
				Console.WriteLine(chars);
			}
			streamReader.Close();


			streamReader = new StreamReader(tempFile);
			string text = "";
			while ((text = streamReader.ReadLine()) != null)
			{
				Console.WriteLine(text);
			}
			streamReader.Close();

			fileStream = new FileStream(tempFile, FileMode.Open);
			byte retVal = (byte) fileStream.ReadByte();

			fileStream = new FileStream(tempFile, FileMode.Open);
			int retByte = 0;
			while ((retByte = fileStream.ReadByte()) != -1)
			{
				Console.WriteLine((byte)retByte);
			}
			fileStream.Close();

			fileStream = new FileStream(tempFile, FileMode.Open);
			long currPosition = 0;
			while (currPosition < fileStream.Length)
			{
				Console.WriteLine((byte) fileStream.ReadByte());
				currPosition++;
			}
			fileStream.Close();

			fileStream = new FileStream(tempFile, FileMode.Open);
			BinaryReader binaryReader = new BinaryReader(fileStream);
			while (binaryReader.PeekChar() != -1)
			{
				Console.WriteLine(binaryReader.ReadChar());
			}
			binaryReader.Close();

		}

		public static FileStream CreateLogFile(string logFileName)
		{
			FileStream fileStream = new FileStream(logFileName, 
				FileMode.Create, 
				FileAccess.ReadWrite, 
				FileShare.None);
			return (fileStream);
		}

        public static void WriteToLog(string logFileName, string data)
        {
            using (FileStream fileStream = new FileStream(logFileName,
                                               FileMode.Append,
                                               FileAccess.Write,
                                               FileShare.None))
            {
                using (StreamWriter streamWriter = new StreamWriter(fileStream))
                {
                    streamWriter.WriteLine(data);
                }
            }
        }

        public static string ReadAllLog(string logFileName)
        {
            if (!File.Exists(logFileName))
            {
                throw (new FileNotFoundException(
                    "Pliku zdarze nie mona otworzy, poniewa nie istnieje.", logFileName));
            }

            string contents = "";

            using (FileStream fileStream = new FileStream(logFileName,
                                               FileMode.Open,
                                               FileAccess.Read,
                                               FileShare.None))
            {
                using (StreamReader streamReader = new StreamReader(fileStream))
                {
                    contents = streamReader.ReadToEnd();
                }
            }

            return contents;
        }

        public static void ReadLogPeeking(string logFileName)
        {
            if (!File.Exists(logFileName))
            {
                throw (new FileNotFoundException(
                    "Pliku zdarze nie mona odczyta, poniewa nie istnieje.", logFileName));
            }

            using (FileStream fileStream = new FileStream(logFileName,
                                               FileMode.Open,
                                               FileAccess.Read,
                                               FileShare.None))
            {
                Console.WriteLine("Odczyt pliku wiersz po wierszu metod Peek:");
                using (StreamReader streamReader = new StreamReader(fileStream))
                {
                    while (streamReader.Peek() != -1)
                    {
                        Console.WriteLine(streamReader.ReadLine());
                    }
                }
            }
        }

        public static void ReadLogByLines(string logFileName)
        {
            if (!File.Exists(logFileName))
            {
                throw (new FileNotFoundException(
                    "Pliku zdarze nie mona odczyta, poniewa nie istnieje.", logFileName));
            }

            using (FileStream fileStream = new FileStream(logFileName,
                                               FileMode.Open,
                                               FileAccess.Read,
                                               FileShare.None))
            {
                Console.WriteLine("Odczyt pliku wiersz po wierszu metod ReadLine:");
                using (StreamReader streamReader = new StreamReader(fileStream))
                {
                    string text = streamReader.ReadLine());
                    while (text != null)
                    {
                        Console.WriteLine(text);
                        text = streamReader.ReadLine();
                    }
                }
            }
        }

        public static void ReadAllLogAsBytes(string logFileName)
        {
            if (!File.Exists(logFileName))
            {
                throw (new FileNotFoundException(
                    "Pliku zdarze nie mona odczyta, poniewa nie istnieje.", logFileName));
            }

            using (FileStream fileStream = new FileStream(logFileName,
                                               FileMode.Open,
                                               FileAccess.Read,
                                               FileShare.None))
            {
                Console.WriteLine("Wczytanie strumienia pliku w postacji bajtw:");
                using (StreamReader streamReader = new StreamReader(fileStream))
                {
                    while (streamReader.Peek() != -1)
                    {
                        Console.Write(streamReader.Read());
                    }
                }
            }
        }

        public static void ReadAllBufferedLog(string logFileName)
        {
            if (!File.Exists(logFileName))
            {
                throw (new FileNotFoundException(
                    "Pliku zdarze nie mona odczyta, poniewa nie istnieje.", logFileName));
            }

            using (FileStream fileStream = new FileStream(logFileName,
                                               FileMode.Open,
                                               FileAccess.Read,
                                               FileShare.None))
            {
                Console.WriteLine("Odczyt strumienia pliku w postaci buforw bajtw:");
                using (StreamReader streamReader = new StreamReader(fileStream))
                {
                    while (streamReader.Peek() != -1)
                    {
                        char[] buffer = new char[10];
                        int bufferFillSize = streamReader.Read(buffer, 0, 10);
                        foreach (char c in buffer)
                        {
                            Console.Write(c);
                        }
                        Console.WriteLine(bufferFillSize);
                    }
                }
            }
        }

		public static void ReadAllBufferedLogBlock(FileStream fileStream)
		{
			if(!fileStream.CanRead)
			{
                // zamknicie i ponowne otwarcie do odczytu
				string fileName = fileStream.Name;
				fileStream.Close();
				fileStream = new FileStream(fileName,FileMode.Open,FileAccess.Read);
			}
            Console.WriteLine("Odczyt strumienia pliku w postaci buforw bajtw za pomoc ReadBlock:");
			StreamReader streamReader = new StreamReader(fileStream);
			while (streamReader.Peek() != -1)
			{
				char[] buffer = new char[10];
				int bufferFillSize = streamReader.ReadBlock(buffer, 0, 10);
				foreach (char c in buffer)
				{
					Console.Write(c);
				}
				Console.WriteLine(bufferFillSize);
			}
			streamReader.Close();
		}

		#endregion

        #region 12.9 Ustalanie, czy istnieje katalog
        public static void DetermineDirectoryExists()
		{
			if (Directory.Exists(@"c:\temp"))
			{
				// Operacje na katalogu
				Console.WriteLine(@"Znaleziono C:\temp");
			}
		}
		#endregion

        #region 12.10 Tworzenie, kopiowanie, przenoszenie i usuwanie katalogu
        public static void CreateMoveDeleteDirectory()
		{
			DirectoryInfo dirInfo = null;
			if (!Directory.Exists(@"c:\delete\test"))
			{
				dirInfo = Directory.CreateDirectory(@"c:\delete\test");
			}

			dirInfo = null;
			if (!Directory.Exists(@"c:\delete\test"))
			{
				dirInfo = new DirectoryInfo(@"c:\delete\test");
				dirInfo.Create();
			}

			if (!Directory.Exists(@"c:\MovedDir"))
			{
				Directory.Move(@"c:\delete", @"c:\MovedDir");
			}

			dirInfo = null;
			if (!Directory.Exists(@"c:\MovedDir"))
			{
				dirInfo = new DirectoryInfo(@"c:\delete\test");
				dirInfo.MoveTo(@"c:\MovedDir");
			}

			if (Directory.Exists(@"c:\MovedDir"))
			{
				Directory.Delete(@"c:\MovedDir", true);
			}

			dirInfo = null;
			if (Directory.Exists(@"c:\MovedDir"))
			{
				dirInfo = new DirectoryInfo(@"c:\delete\test");
				dirInfo.Delete(true);
			}

		}
		#endregion

        #region 12.11 Operacje na atrybutach katalogw
        public static void ManipulateDirectoryAttribs()
		{
			string path = Path.GetTempPath() + @"\MyTemp";
			Directory.CreateDirectory(path);
			try
			{
				// Wywietlanie i operacje na znacznikach czasu i atrybutach katalogu
                // przy uyciu statycznych metod klasy Directory.
				DisplayDirAttr1(path);
				ModifyDirAttr1(path);


				// Wykorzystanie metod egzemplarza klasy DirectoryInfo do operowania
                // na znacznikach czasu i atrybutach katalogu.
				DisplayDirAttr2(path);
				ModifyDirAttr2(path);
				ViewModifyDirAttr(path,FileAttributes.Hidden);
			}
			catch(Exception e)
			{
				Console.WriteLine("{0}",e.ToString());
			}
			Directory.Delete(path,true);
		}

		public static void DisplayDirAttr1(string path)
		{
			Console.WriteLine(Directory.GetCreationTime(path).ToString());
			Console.WriteLine(Directory.GetLastAccessTime(path).ToString());
			Console.WriteLine(Directory.GetLastWriteTime(path).ToString());
		}

		public static void DisplayDirAttr2(string path)
		{
			DirectoryInfo dirInfo = Directory.CreateDirectory(path);

			Console.WriteLine(dirInfo.CreationTime.ToString());
			Console.WriteLine(dirInfo.LastAccessTime.ToString());
			Console.WriteLine(dirInfo.LastWriteTime.ToString());
		}

		public static void ModifyDirAttr1(string path)
		{
			DateTime dt = new DateTime(2003,5,10);
			Directory.SetCreationTime(path, dt);
			Directory.SetLastAccessTime(path, dt);
			Directory.SetLastWriteTime(path, dt);
		}

		public static void ModifyDirAttr2(string path)
		{
			DirectoryInfo dirInfo = Directory.CreateDirectory(path);

			DateTime dt = new DateTime(2001,2,8);
			dirInfo.CreationTime = dt;
			dirInfo.LastAccessTime = dt;
			dirInfo.LastWriteTime = dt; 
		}

		public static void ViewModifyDirAttr(string path, FileAttributes fileAttributes)
		{
			DirectoryInfo dirInfo = new DirectoryInfo(@"C:\Windows\System32"); 
			// wywietlenie atrybutw katalogu
			Console.WriteLine(dirInfo.Attributes);

			// wywietlenie informacji, czy katalog jest ukryty
			Console.WriteLine("Czy katalog jest ukryty? = " + 
				((dirInfo.Attributes & FileAttributes.Hidden) == FileAttributes.Hidden));

			// zmiana atrybutw katalogu
			dirInfo.Attributes |= fileAttributes;

            // wywietlenie informacji, czy katalog jest ukryty
            Console.WriteLine("Czy katalog jest ukryty? = " + 
				((dirInfo.Attributes & FileAttributes.Hidden) == FileAttributes.Hidden));
		}
		#endregion

        #region 12.12 Zmiana nazwy katalogu
        public static void RenameDirectory()
		{
			string path = Path.GetTempPath() + "MyTemp";
			string newpath = path + "1";
			Directory.CreateDirectory(path);

			DemonstrateRenameDir1(path,newpath);
			DemonstrateRenameDir2(newpath,path);
		}

		public static void DemonstrateRenameDir1(string originalName, string newName)
		{
			try
			{
				Directory.CreateDirectory(originalName);
				// "zmiana nazwy"
				Directory.Move(originalName, newName);
			}
			catch(IOException ioe) 
			{
				// najprawdopodobniej katalog ju istnieje albo nie jest pusty
				Console.WriteLine(ioe.ToString());
			}
			catch(Exception e) 
			{
				// obsuga wszystkich innych wyjtkw
				Console.WriteLine(e.ToString());
			} 
		}

		public static void DemonstrateRenameDir2 (string originalName, string newName)
		{
			try
			{
				DirectoryInfo dirInfo = new DirectoryInfo(originalName);
				// utworzenie katalogu
				dirInfo.Create();
				// "zmiana nazwy"
				dirInfo.MoveTo(newName);
			}
			catch(IOException ioe) 
			{
                // najprawdopodobniej katalog ju istnieje albo nie jest pusty
				Console.WriteLine(ioe.ToString());
			}
			catch(Exception e) 
			{
                // obsuga wszystkich innych wyjtkw
				Console.WriteLine(e.ToString());
			} 
		}

		#endregion

        #region 12.13 Wyszukiwanie katalogw lub plikw przy uyciu symboli wieloznacznych
        public static void SearchDirFileWildcards()
		{
			try
			{
				// zdefiniowanie cieek pocztkowych
				string tempPath = Path.GetTempPath();		
				string tempDir = tempPath + @"\MyTempDir";
				string tempDir2 = tempDir + @"\MyNestedTempDir";
				string tempDir3 = tempDir + @"\MyNestedTempDirPattern";
				string tempFile = tempDir + @"\MyTempFile.PDB";
				string tempFile2 = tempDir2 + @"\MyNestedTempFile.PDB";
				string tempFile3 = tempDir + @"\MyTempFile.TXT";

				// utworzenie plikw i katalogw tymczasowych
				Directory.CreateDirectory(tempDir);
				Directory.CreateDirectory(tempDir2);
				Directory.CreateDirectory(tempDir3);
				FileStream stream = File.Create(tempFile);
				FileStream stream2 = File.Create(tempFile2);
				FileStream stream3 = File.Create(tempFile3);
				// zamknicie plikw przed uyciem
				stream.Close();
				stream2.Close();
				stream3.Close();

				// wywietlenie wszystkich elementw w katalogu tymczasowym
				DisplayFilesDirs1(tempDir);

				// wywietlenie elementw pasujcych do wzorca
				string pattern = "*.PDB";
				DisplayFilesDirs3(tempDir,pattern);

				// wywietlenie katalogw na ciece dostpu
				DisplayDirs1(tempDir);

				pattern = "*Pattern*";
				DisplayDirs3(tempDir,pattern);
				
				// wywietlenie wszystkich plikw
				DisplayFiles1(tempDir);

                // wywietlenie wszystkich plikw pasujcych do wzorca
				pattern = "*.PDB";
				DisplayFiles3(tempDir,pattern);

				// Metody, ktre zwracaj prawdziwe obiekty zamiast cieek dostpu
				
				// odczytanie elementw
				DisplayFilesDirs2(tempDir);

				// odczytanie elementw pasujcych do wzorca
				pattern = "*.PDB";
				DisplayFilesDirs4(tempDir,pattern);

				// odczytanie informacji o katalogu
				DisplayDirs2(tempDir);

				// odczytanie informacji o katalogach pasujcych do wzorca
				pattern = "*Pattern*";
				DisplayDirs4(tempDir,pattern);

				// odczytanie informacji o pliku
				DisplayFiles2(tempDir);

                // odczytanie informacji o plikach pasujcych do wzorca
				pattern = "*.PDB";
				DisplayFiles4(tempDir,pattern);

				// wyczyszczenie elementw tymczasowych
				File.Delete(tempFile3);
				File.Delete(tempFile2);
				File.Delete(tempFile);
				Directory.Delete(tempDir3);
				Directory.Delete(tempDir2);
				Directory.Delete(tempDir);
			}
			catch(Exception e)
			{
				Console.WriteLine(e.ToString());
			}
		}

		public static void DisplayFilesDirs1(string path)
		{
			string[] items = Directory.GetFileSystemEntries(path);
			foreach (string item in items)
			{
				Console.WriteLine(item);
			}
		}
		
		public static void DisplayDirs1(string path)
		{
			string[] items = Directory.GetDirectories(path);
			foreach (string item in items)
			{
				Console.WriteLine(item);
			}
		}

		public static void DisplayFiles1(string path)
		{
			string[] items = Directory.GetFiles(path);
			foreach (string item in items)
			{
				Console.WriteLine(item);
			}
		}

		public static void DisplayFilesDirs2(string path)
		{
			DirectoryInfo mainDir = new DirectoryInfo(path);
			FileSystemInfo[] items = mainDir.GetFileSystemInfos();
			foreach (FileSystemInfo item in items)
			{
				if (item is DirectoryInfo)
				{
					Console.WriteLine("KATALOG: " + ((DirectoryInfo)item).Name);
				}
				else if (item is FileInfo)
				{
					Console.WriteLine("PLIK: " + item.Name);
				}
				else
				{
					Console.WriteLine("Nieznany");
				}
			}
		}

		public static void DisplayDirs2(string path)
		{
			DirectoryInfo mainDir = new DirectoryInfo(path);
			DirectoryInfo[] items = mainDir.GetDirectories();
			foreach (DirectoryInfo item in items)
			{
				Console.WriteLine("KATALOG: " + ((DirectoryInfo)item).Name);
			}
		}

		public static void DisplayFiles2(string path)
		{
			DirectoryInfo mainDir = new DirectoryInfo(path);
			FileInfo[] items = mainDir.GetFiles();
			foreach (FileInfo item in items)
			{
				Console.WriteLine("PLIK: " + item.Name);
			}
		}

		// przekazanym wzorcem jest cig znakw rwny "*.PDB"
		public static void DisplayFilesDirs3(string path, string pattern)
		{
			string[] items = Directory.GetFileSystemEntries(path, pattern);
			foreach (string item in items)
			{
				Console.WriteLine(item);
			}
		}

        // przekazanym wzorcem jest cig znakw rwny "*.PDB"
		public static void DisplayDirs3(string path, string pattern)
		{
			string [] items = Directory.GetDirectories(path, pattern);
			foreach (string item in items)
			{
				Console.WriteLine(item);
			}
		}

        // przekazanym wzorcem jest cig znakw rwny "*.PDB"
		public static void DisplayFiles3(string path, string pattern)
		{
			string [] items = Directory.GetFiles(path, pattern);
			foreach (string item in items)
			{
				Console.WriteLine(item);
			}
		}

        // przekazanym wzorcem jest cig znakw rwny "*.PDB"
		public static void DisplayFilesDirs4(string path, string pattern)
		{
			DirectoryInfo mainDir = new DirectoryInfo(path);
			FileSystemInfo[] items = mainDir.GetFileSystemInfos(pattern);
			foreach (FileSystemInfo item in items)
			{
				if (item is DirectoryInfo)
				{
					Console.WriteLine("KATALOG: " + ((DirectoryInfo)item).Name);
				}
				else if (item is FileInfo)
				{
					Console.WriteLine("PLIK: " + item.Name);
				}
				else
				{
					Console.WriteLine("Nieznany");
				}
			}
		}

        // przekazanym wzorcem jest cig znakw rwny "*.PDB"
		public static void DisplayDirs4(string path, string pattern)
		{
			DirectoryInfo mainDir = new DirectoryInfo(@"C:\TEMP ");
			DirectoryInfo[] items = mainDir.GetDirectories("*.PDB");
			foreach (DirectoryInfo item in items)
			{
				Console.WriteLine("KATALOG: " + ((DirectoryInfo)item).Name);
			}
		}

        // przekazanym wzorcem jest cig znakw rwny "*.PDB"
		public static void DisplayFiles4(string path, string pattern)
		{
			DirectoryInfo mainDir = new DirectoryInfo(@"C:\TEMP ");
			FileInfo[] items = mainDir.GetFiles("*.PDB");
			foreach (FileInfo item in items)
			{
				Console.WriteLine("PLIK: " + item.Name);
			}
		}

		#endregion

        #region 12.14 Odczytywanie drzewa katalogw
        public static void ObtainDirTree()
		{
			try
			{
				// utworzenie elementw podlegajcych wyszukiwaniu...
				string tempPath = Path.GetTempPath();		
				string tempDir = tempPath + @"\MyTempDir";
				string tempDir2 = tempDir + @"\Chapter 1 - The Beginning";
				string tempDir3 = tempDir2 + @"\Chapter 1a - The Rest Of The Beginning";
				string tempDir4 = tempDir2 + @"\IHaveAPDBFile";
				string tempFile = tempDir + @"\MyTempFile.PDB";
				string tempFile2 = tempDir2 + @"\MyNestedTempFile.PDB";
				string tempFile3 = tempDir3 + @"\Chapter 1 - MyDeepNestedTempFile.PDB";
				string tempFile4 = tempDir4 + @"\PDBFile.PDB";
				// Drzewo ma nastpujc posta
				// TEMP\
				//		MyTempDir\
				//				Chapter 1 - The Beginning\
				//								Chapter 1a - The Rest Of The Beginning\
				//								IHaveAPDBFile
				//

				// utworzenie tymczasowych plikw i katalogw
				Directory.CreateDirectory(tempDir);
				Directory.CreateDirectory(tempDir2);
				Directory.CreateDirectory(tempDir3);
				Directory.CreateDirectory(tempDir4);
				FileStream stream = File.Create(tempFile);
				FileStream stream2 = File.Create(tempFile2);
				FileStream stream3 = File.Create(tempFile3);
				FileStream stream4 = File.Create(tempFile4);
				// zamknicie plikw przed ich wykorzystaniem
				stream.Close();
				stream2.Close();
				stream3.Close();
				stream4.Close();

				// odczytanie listy wszystkich katalogw
				GetAllDirFilesRecurse(tempDir);

				// odczytanie listy wszystkich plikw z rozszerzeniem PDB, that 
				// ktre rozpoczynaj sie od Rozdzia 1:
				GetAllFilesInPatternRecurse(tempDir);

				// usunicie elementw tymczasowych
				File.Delete(tempFile4);
				File.Delete(tempFile3);
				File.Delete(tempFile2);
				File.Delete(tempFile);
				Directory.Delete(tempDir4);
				Directory.Delete(tempDir3);
				Directory.Delete(tempDir2);
				Directory.Delete(tempDir);
			}
			catch(Exception e)
			{
				Console.WriteLine(e.ToString());
			}
		}

		public static void GetAllDirFilesRecurse(string dir)
		{
			DirectoryInfo mainDir = new DirectoryInfo(dir);
			FileSystemInfo[] items = mainDir.GetFileSystemInfos();
			foreach (FileSystemInfo item in items)
			{
				if (item is DirectoryInfo)
				{
					Console.WriteLine("KATALOG: " + ((DirectoryInfo)item).FullName);
					GetAllDirFilesRecurse(((DirectoryInfo)item).FullName);
				}
				if (item is FileInfo)
				{
					Console.WriteLine("PLIK: " + ((FileInfo)item).FullName);
				}
				else
				{
					Console.WriteLine("Nieznany");
				}
			} 
		}

		public static void GetAllFilesInPatternRecurse(string dir)
		{
			DirectoryInfo mainDir = new DirectoryInfo(dir);
			FileSystemInfo[] items = mainDir.GetFileSystemInfos("Rozdzia 1*");
			foreach (FileSystemInfo item in items)
			{
				if (item is DirectoryInfo)
				{
					GetAllFilesInPatternRecurse(((DirectoryInfo)item).FullName);
				}
				if (item is FileInfo)
				{
					FileInfo fileInfo = item as FileInfo;
					if(fileInfo.Extension.ToUpper().CompareTo(".PDB")==0)
						Console.WriteLine("PLIK: " + (fileInfo.FullName));
				}
			}
		}
		#endregion

        #region 12.15 Parsowanie cieki dostpu
        public static void ParsePath()
		{
			ParsePath(@"c:\test\tempfile.txt");
		}

		public static void ParsePath(string path)
		{
			string root = Path.GetPathRoot(path);
			string dirName = Path.GetDirectoryName(path);
			string fullFileName = Path.GetFileName(path);
			string fileExt = Path.GetExtension(path);
			string fileNameWithoutExt = Path.GetFileNameWithoutExtension(path);
			StringBuilder format = new StringBuilder();
            format.Append("W wyniku parsowania cieki {0} metod ParsePath uzyskano nastpujce elementy:\r\n\tKatalog gwny: {1}\r\n\t");
			format.Append("Nazwa katalogu: {2}\r\n\tPena nazwa pliku: {3}\r\n\t");
			format.Append("Rozszerzenie pliku: {4}\r\n\tNazwa pliku bez rozszerzenia: {5}\r\n");
			Console.WriteLine(format.ToString(),path,root,dirName,fullFileName,fileExt,fileNameWithoutExt);
		}

		#endregion

        #region 12.16 Parsowanie cieek dostpu w zmiennych rodowiskowych
        public static void ParsePathsEnvVars()
		{
			ParsePathEnvironmentVariable();
		}

		public static void ParsePathEnvironmentVariable()
		{
			string originalPathEnv = Environment.GetEnvironmentVariable("PATH");
			string[] paths = originalPathEnv.Split(new char[1] {Path.PathSeparator});
			foreach (string s in paths)
			{
				string pathEnv = Environment.ExpandEnvironmentVariables(s);
                Console.WriteLine("cieka = " + pathEnv); 
                if (pathEnv.Length > 0)
				{
					Console.WriteLine("Pojedyncza cieka = " + pathEnv);
				}
				else
				{
					Console.WriteLine("Zignorowano pust ciek, " +
						" poniewa jest ona przyczyn wyjtkw...");
													}
				Console.WriteLine();
			}
		}

		#endregion

        #region 12.17 Weryfikacja cieki dostpu
        public static void VerifyPath()
		{
			CheckUserEnteredPath(@"Q:\MyNonExistentPath\MyNonExistentFile.txt", true);
			CheckUserEnteredPath(@"Q:\MyNonExistentPath\MyNonExistentFile.txt", false);
			CheckUserEnteredPath(@"Q:\MyNonExistentPath\", true);
			CheckUserEnteredPath(@"Q:\MyNonExistentPath\", false);
			CheckUserEnteredPath(@"Q:\MyNonExistentPath\MyNonExistentFile", true);
			CheckUserEnteredPath(@"Q:\MyNonExistentPath\MyNonExistentFile", false);
			CheckUserEnteredPath(@"Q:\<MyNonExistentPath>\MyNonExistentFile.txt", true);
			CheckUserEnteredPath(@"Q:\<MyNonExistentPath>\MyNonExistentFile.txt", false);
			CheckUserEnteredPath(@"Q:\MyNonExistentPath\MyNonExist:entFil*e.txt", true);
			CheckUserEnteredPath(@"Q:\MyNonExistentPath\MyNonExist:entFil*e.txt", false);
			CheckUserEnteredPath(null, true);
		}

		public static bool CheckUserEnteredPath(string path, bool any)
		{
			try
			{
				Console.WriteLine("Sprawdzanie cieki {0}",path);

				// sprawdzenie, czy cieka nie jest pusta
				if (path == null)
				{
					throw (new ArgumentNullException("path", "Wpisana cieka nie moe by pusta"));
				}
				
				bool illegal = false;
                List<char> invalidChars = new List<char>();

				// dwa sposoby wyszukiwania: jeden szybszy, drugi wolniejszy
				if(any == true)
					illegal = FindAnyIllegalChars(path);    // szybszy
				else
					invalidChars = FindAllIllegalChars(path);    // wolniejszy

				if (!illegal && invalidChars.Count == 0)
				{
					// upewnienie si, e cieka nie jest pustym cigiem znakw
					// i zawiera nazw pliku i rozszerzenie
					if (Path.GetFileName(path).Length == 0)
					{
						Console.WriteLine("Naley poda nazw pliku");
					}    
					else if (!Path.HasExtension(path))
					{
						Console.WriteLine("Nazwa pliku musi posiada rozszerzenie");
					}
					else
					{
						Console.WriteLine("cieka jest prawidowa");
						return (true);
					}
                }
                else if (invalidChars.Count > 0)
                {
                    foreach(char c in invalidChars)
                        Console.WriteLine(c);
                }				
			}
			catch(Exception e)
			{    
				Console.WriteLine(e.ToString());
			}
			
			return (false);
		}

		private static bool FindAllIllegalChars(string path)
		{
			// odczytanie katalogu podanego w ciece
			string dirName = path;
			string fullFileName = "";
            int pos = path.LastIndexOf(Path.DirectorySeparatorChar);
			if (pos >= 0)
			{
				dirName = path.Substring(0, pos);

				// odczytanie nazwy pliku podanego w ciece
				if (pos >= 0 && (pos + 1) < path.Length)
					fullFileName = path.Substring(pos + 1);
			}
			
			int  invalidCharPos = 0;
			bool endOfPath = false;
			bool foundIllegalChars = false;

			// wyszukanie wszystkich niedozwolonych znakw w nazwie katalogu
			while (!endOfPath)
			{
				invalidCharPos = dirName.IndexOfAny(Path.GetInvalidPathChars(), 
					invalidCharPos++);
				if (invalidCharPos == -1)
				{
					endOfPath = true;
				}
				else
				{
					foundIllegalChars = true;
					Console.WriteLine("Niedozwolony znak {0} znajduje si w ciece katalogw na pozycji {1}.", dirName[invalidCharPos], invalidCharPos);
                    invalidChars.Add(dirName[invalidCharPos]);

                    if (invalidCharPos >= dirName.Length - 1)
					{
						endOfPath = true;
					}
					else
					{
						invalidCharPos++;
					}
				}
			}

			bool endOfFileName = false;
			invalidCharPos = 0;

			// wyszukanie w nazwie plikw wszystkich niedozwolonych znakw
			while (!endOfFileName)
			{
				invalidCharPos = fullFileName.IndexOfAny(Path.GetInvalidFileNameChars(),
					invalidCharPos++);
				if (invalidCharPos == -1)
				{
					endOfFileName = true;
				}
				else
				{
					foundIllegalChars = true;
                    Console.WriteLine("Niedozwolony znak {0} znajduje si w nazwie pliku na pozycji {1}.", fullFileName[invalidCharPos], invalidCharPos);
                    invalidChars.Add(fullFileName[invalidCharPos]);

					if (invalidCharPos >= fullFileName.Length - 1)
					{
						endOfFileName = true;
					}
					else
					{
						invalidCharPos++;
					}
				}
			}

			return (foundIllegalChars);
 		}

		private static bool FindAnyIllegalChars(string path)
		{
            // odczytanie katalogu podanego w ciece
			string dirName = path;
			string fullFileName = "";
            int pos = path.LastIndexOf(Path.DirectorySeparatorChar);
			if (pos >= 0)
			{
				dirName = path.Substring(0, pos);

                // odczytanie nazwy pliku podanego w ciece
				if (pos >= 0 && (pos + 1) < path.Length)
					fullFileName = path.Substring(pos + 1);
			}

            // wyszukanie dowolnego niedozwolonego znaku w nazwie katalogu
			int invalidCharPos = dirName.IndexOfAny(Path.GetInvalidPathChars());
			if (invalidCharPos == -1)
			{
                // wyszukanie dowolnego niedozwolonego znaku w nazwie pliku
				invalidCharPos = fullFileName.IndexOfAny(Path.GetInvalidFileNameChars());
				if (invalidCharPos == -1)
				{
					return (false);
				}
				else
				{
                    Console.WriteLine("Niedozwolony znak {0} znajduje si w nazwie pliku na pozycji {1}.", fullFileName[invalidCharPos], invalidCharPos);
					return (true);
				}
			}
			else
			{
                Console.WriteLine("Niedozwolony znak {0} znajduje si w ciece katalogw na pozycji {1}.", dirName[invalidCharPos], invalidCharPos);
				return (true);
			}
		}

		#endregion

        #region 12.18 Uywanie w aplikacji pliku tymczasowego
        public static void UseTempFile()
		{
			try
			{
				// wywietlenie cieki katalogu tymczasowego
				string tempDir = Path.GetTempPath();
				Console.WriteLine("Katalog TEMP znajduje si w {0}",tempDir);
				// utworzenie pustego pliku tymczasowego w katalogu tymczasowym
				string tempFile = Path.GetTempFileName();
				Console.WriteLine("Utworzono plik tymczasowy {0}",tempFile);
				// usunicie pliku tymczasowego
				File.Delete(tempFile);
			}
			catch(Exception e)
			{
				Console.WriteLine(e.ToString());
			}
		}
		#endregion

        #region 12.19 Otwieranie strumienia pliku przy uyciu jedynie uchwytu pliku
        public static void OpenFileStreamWithFileHandle()
		{
			UsingAnUnmanagedFileHandle();
		}

		public static void UsingAnUnmanagedFileHandle()
		{
			IntPtr hFile = IntPtr.Zero;
			// utworzenie pliku w kodzie niezarzdzanym
			hFile = (IntPtr)FileInteropFunctions.CreateFile("data.txt",
				FileInteropFunctions.GENERIC_WRITE,
				0,
				IntPtr.Zero,
				FileInteropFunctions.CREATE_ALWAYS,
				0,
				0);

			if(hFile.ToInt64() > 0)
			{
				// zapis do pliku w kodzie niezarzdzanym
				WriteToFileHandle(hFile);
				// zamknicie pliku
				FileInteropFunctions.CloseHandle(hFile);
				// usunicie pliku
				File.Delete("data.txt");
			}
		}

		public static void WriteToFileHandle(IntPtr hFile)
		{
			// umieszczenie uchwytu pliku w obiekcie z bezpieczn obsug
			Microsoft.Win32.SafeHandles.SafeFileHandle safeHFile = 
				new Microsoft.Win32.SafeHandles.SafeFileHandle(hFile, true);
			
			// otwarcie obiektu FileStream za pomoc przekazanego bezpiecznego uchwytu pliku
			FileStream fileStream = new FileStream(safeHFile, FileAccess.ReadWrite);
			// oprnienie strumienia przed rozpoczciem wykonywania akcji niezarzdzanych
			fileStream.Flush();    
			// operacje na pliku...
			string line = "Ten wiersz zosta zapisany przez kod zarzdzany!";
			// zapis do pliku
			byte[] bytes = Encoding.ASCII.GetBytes(line);
			fileStream.Write(bytes,0,bytes.Length);
			// zamknicie strumienia pliku
			fileStream.Close();
		}

		class FileInteropFunctions
		{
			public const uint GENERIC_READ = (0x80000000);
			public const uint GENERIC_WRITE = (0x40000000);
			public const uint GENERIC_EXECUTE = (0x20000000);
			public const uint GENERIC_ALL = (0x10000000);

			public const uint CREATE_NEW        = 1;
			public const uint CREATE_ALWAYS     = 2;
			public const uint OPEN_EXISTING     = 3;
			public const uint OPEN_ALWAYS       = 4;
			public const uint TRUNCATE_EXISTING = 5;

			[DllImport("kernel32.dll", SetLastError=true)]
			public static extern bool CloseHandle(IntPtr hObject);        

			[DllImport("kernel32.dll", SetLastError=true)]
			public static extern IntPtr CreateFile(
				String lpFileName,              // nazwa pliku
				uint dwDesiredAccess,              // tryb dostpu
				uint dwShareMode,              // tryb wspuytkowania
				IntPtr attr,                  // deskryptor bezpieczestwa
				uint dwCreationDisposition,          // sposb utworzenia
				uint dwFlagsAndAttributes,          // atrybuty pliku
                uint hTemplateFile);          // uchwyt pliku szablonu

			[DllImport("kernel32.dll", SetLastError=true)]
			public static extern bool FlushFileBuffers(IntPtr hFile);
		}

		#endregion

        #region 12.20 Jednoczesne zapisywanie do wielu plikw wyjciowych
        public static void WriteMultipleOutputFiles()
		{
			// utworzenie listy trzech nazw plikw
			string[] names = new string[3];
			for (int i=0;i<3;i++)
			{
				names[i] = Path.GetTempFileName();
			}
			MultiWriter multi = new MultiWriter(names);
			multi.WriteLine("Pierwszy wiersz");
			multi.WriteLine("Drugi wiersz");
			multi.WriteLine("Trzeci wiersz");
			multi.Close();
		}

		class MultiWriter : IDisposable
		{
			FileStream[] _streams;
			string [] _names;
			int _streamCount = 0;
			bool _disposed = false;

			public MultiWriter(string[] fileNames)
			{
				try
				{
					// skopiowanie nazw
					_names = (string[])fileNames.Clone();
					// ustawienie liczby strumieni
					_streamCount = fileNames.Length;
					// utworzenie tablicy strumieni
					_streams = new FileStream[_streamCount];
					for(int i = 0; i < _streams.Length; i++)
					{
						// utworzenie strumienia pliku
						_streams[i] = new FileStream(_names[i], 
							FileMode.Create, 
							FileAccess.ReadWrite, 
							FileShare.None);
					}
				}
				catch(IOException ioe)
				{
					Console.WriteLine(ioe.ToString());
				}
			}

			public void WriteLine(string text)
			{
				// dodanie nowego wiersza
				text += Environment.NewLine;
				// odczytanie bajtw w formacie unicode...
				byte[] bytes = Encoding.ASCII.GetBytes(text);
				// przejcie przez strumienie
				for(int i = 0; i < _streams.Length; i++)
				{
					// zapisanie tekstu
					_streams[i].Write(bytes,0,bytes.Length);
				}
			}

			public void Close()
			{
				Dispose();
			}

            public void Dispose()
            {
                Dispose(true);
                // zapobieenie ponownej finalizacji
                GC.SuppressFinalize(this);
            }

			protected void Dispose()
			{
				try
				{
					// zamknicie tylko jeden raz
					if(_disposed == false)
					{
						// zamknicie kadego strumienia
						for(int i=0;i<_streams.Length;i++)
						{
							_streams[i].Close();
						}
						
						// wskazanie, e zamknicie zostao ju dokonane
						_disposed = true;
					}
				}
				catch(IOException ioe)
				{
					Console.WriteLine(ioe.ToString());
				}
			}

            ~MultiWriter()
            {
                Dispose(false);
            }
		}
		#endregion

        #region 12.21 Uruchamianie i uywanie narzdzi konsoli
        public static void LaunchInteractConsoleUtils()
		{
			RunProcessToReadStdIn();
		}

		public static void RunProcessToReadStdIn()
		{
			Process application = new Process();
			// uruchomienie powoki
			application.StartInfo.FileName = @"cmd.exe";

			// wczenie standardowych rozszerze
			application.StartInfo.Arguments = "/E:ON";

			application.StartInfo.RedirectStandardInput = true;

			application.StartInfo.UseShellExecute = false;

			// uruchomienie procesu
			application.Start();

			// pobranie standardowego wejcia
			StreamWriter input = application.StandardInput;

			// wykonanie polecenia wywietlajcego czas
			input.WriteLine("TIME /T");

			// zatrzymanie uruchomionej aplikacji
			input.WriteLine("exit");
		}

		#endregion

        #region 12.22 Blokowanie czci pliku
        public static void LockSubsectionsOfAFile()
		{
			string path = Path.GetTempFileName();
			CreateLockedFile(path);
			File.Delete(path);
		}

        public static void CreateLockedFile(string fileName)
        {
            using (FileStream fileStream = new FileStream(fileName,
                FileMode.Create,
                FileAccess.ReadWrite,
                FileShare.ReadWrite))
            {
                using (StreamWriter streamWriter = new StreamWriter(fileStream))
                {
                    streamWriter.WriteLine("Pierwszy wiersz");
                    streamWriter.WriteLine("Drugi wiersz");
                    streamWriter.Flush();

                    // zablokowanie caego pliku
                    fileStream.Lock(0, fileStream.Length);

                    // wykonanie duszego przetwarzania...
                    Thread.Sleep(1000);

                    // Trzeba zapewni, e plik zostanie odblokowany.            
                    // Jeli proces si zakoczy i cz pliku pozostanie zablokowana lub zamknity zostanie
                    // plik posiadajcy blokady, ktre nie zostay usunite, wwczas zachowanie
                    // moe by nieokrelone. A to w nomenklaturze MS nie oznacza nic dobrego...
                    fileStream.Unlock(0, fileStream.Length);

                    streamWriter.WriteLine("Trzeci wiersz");
                    streamWriter.Close();
                    fileStream.Close();
                }
            }
        }


		#endregion

        #region 12.23 Wyszukiwanie w systemie plikw konkretnych zmian w jednym lub wicej plikach lub katalogach
        public static void WatchFileSystemChanges()
		{
			TestWatcher();
		}

        public static void TestWatcher()
        {
            using (FileSystemWatcher fsw = new FileSystemWatcher())
            {
                fsw.Path = @"c:\";
                fsw.Filter = @"*.txt";
                fsw.IncludeSubdirectories = true;

                fsw.NotifyFilter = NotifyFilters.FileName |
                    NotifyFilters.Attributes |
                    NotifyFilters.LastAccess |
                    NotifyFilters.LastWrite |
                    NotifyFilters.Security |
                    NotifyFilters.Size |
                    NotifyFilters.CreationTime |
                    NotifyFilters.DirectoryName;

                fsw.Changed += new FileSystemEventHandler(OnChanged);
                fsw.Created += new FileSystemEventHandler(OnCreated);
                fsw.Deleted += new FileSystemEventHandler(OnDeleted);
                fsw.Renamed += new RenamedEventHandler(OnRenamed);
                fsw.Error += new ErrorEventHandler(OnError);

                fsw.EnableRaisingEvents = true;

                string file = @"c:\myfile.txt";
                string newfile = @"c:\mynewfile.txt";

                using (FileStream stream = File.Create(file))
                {
                    // uycie zmiennej strumienia...
                    byte[] bytes = new byte[5] { 32, 33, 34, 35, 36 };
                    stream.Write(bytes, 0, bytes.Length);
                }

                using (FileStream stream = File.Create(newfile))
                {
                    // uycie zmiennej strumienia...
                    byte[] bytes = new byte[5] { 32, 33, 34, 35, 36 };
                    stream.Write(bytes, 0, bytes.Length);
                }

                File.Delete(file);
                File.Delete(newfile);

                // moment na przechwycenie przez procedury obsugi zdarze
                // zdarze zachodzcych w systemie plikw
                Thread.Sleep(1000);
            }
        }


		public static void OnChanged(object source, FileSystemEventArgs e) 
		{
			Console.WriteLine("Plik " + e.FullPath + " --> " + e.ChangeType.ToString());
		}

		public static void OnDeleted(object source, FileSystemEventArgs e) 
		{
            Console.WriteLine("Plik " + e.FullPath + " --> " + e.ChangeType.ToString());
		}

		public static void OnCreated(object source, FileSystemEventArgs e) 
		{
            Console.WriteLine("Plik " + e.FullPath + " --> " + e.ChangeType.ToString());
		}

		public static void OnRenamed(object source, RenamedEventArgs e) 
		{
            Console.WriteLine("Plik " + e.OldFullPath + " (zmieni nazw na)--> " + e.FullPath);
		}


		public static void OnError(object source, ErrorEventArgs e) 
		{
			Console.WriteLine("Bd " + e.ToString());
		}

		#endregion

        #region 12.24 Oczekiwanie na wykonanie okrelonej czynnoci w systemie plikw
        public static void WaitFileSystemAction()
		{
			string tempPath = Path.GetTempPath();
			string fileName = "Backup.zip";
			File.Delete(tempPath + fileName);
			
			WaitForZipCreation(tempPath,fileName);
		}

		public static void WaitForZipCreation(string path, string fileName)
		{
			FileSystemWatcher fsw = null;
            try
            {
                using (fsw = new FileSystemWatcher())
                {
                    string[] data = new string[] { path, fileName };
                    fsw.Path = path;
                    fsw.Filter = fileName;
                    fsw.NotifyFilter = NotifyFilters.LastAccess | NotifyFilters.LastWrite
                        | NotifyFilters.FileName | NotifyFilters.DirectoryName;

                    // Wykonanie kodu, ktry tworzy oczekiwany plik
                    // Standardowo plik jest tworzony przez inny proces
                    if (ThreadPool.QueueUserWorkItem(new WaitCallback(PauseAndCreateFile),
                        data))
                    {                        
                        // zablokowanie oczekiwania na zmian
                        WaitForChangedResult result =
                            fsw.WaitForChanged(WatcherChangeTypes.Created);
                        Console.WriteLine("{0} utworzony w {1}.", result.Name, path);
                    }
                }
            }
            catch (Exception e)
            {
                Console.WriteLine(e.ToString());
                throw;
            }
			
			// czyszczenie
			File.Delete(fileName); 
		}

		static void PauseAndCreateFile(Object stateInfo) 
		{
			try
			{
				string[] data = (string[])stateInfo;
				// sekunda...
				Thread.Sleep(1000);
				// utworzenie pliku w katalogu tymczasowym
				string path = data[0];
				string file = path + data[1];
				Console.WriteLine("Tworzenie {0} w metodzie PauseAndCreateFile...",file);
                using (FileStream fileStream = File.Create(file))
                {
                    // uycie zmiennej fileStream...
                }				
			}
			catch(Exception e)
			{
				Console.WriteLine(e.ToString());
                throw;
			}
		}
		#endregion

        #region 12.25 Porwnywanie wersji dwch moduw wykonywalnych
        public static void CompareVersionInfo()
		{
			// Porwnanie wersji
			string file1 = Path.GetTempFileName();
			string file2 = Path.GetTempFileName();
			FileSystemIO.FileComparison result = 
				FileSystemIO.CompareFileVersions(file1,file2);
			// porzdki
			File.Delete(file1);
			File.Delete(file2);
		}

		public enum FileComparison
		{
			Error = 0,
			File1IsNewer = 1,
			File2IsNewer = 2,
			FilesAreSame = 3
		}

		public static FileComparison CompareFileVersions(string file1, string file2)
		{
			FileComparison retValue = FileComparison.Error;
			// czy obydwa pliki istniej?
			if (!File.Exists(file1))
			{
				Console.WriteLine(file1 + " nie istnieje");
			}
			else if (!File.Exists(file2))
			{
                Console.WriteLine(file2 + " nie istnieje");
			}
			else
			{
				// odczytanie wersji
				FileVersionInfo file1Version = FileVersionInfo.GetVersionInfo(file1);
				FileVersionInfo file2Version = FileVersionInfo.GetVersionInfo(file2);

				// sprawdzenie gwnego numeru wersji
				if (file1Version.FileMajorPart > file2Version.FileMajorPart)
				{
					Console.WriteLine(file1 + " jest nowsz wersj");
					retValue = FileComparison.File1IsNewer;
				}
				else if (file1Version.FileMajorPart < file2Version.FileMajorPart)
				{
                    Console.WriteLine(file2 + " jest nowsz wersj");
					retValue = FileComparison.File2IsNewer;
				}
				else // gwne numery wersji s rwne, sprawdzamy dalej...
				{
					// sprawdzenie pobocznego numeru wersji
					if (file1Version.FileMinorPart > file2Version.FileMinorPart)
					{
                        Console.WriteLine(file1 + " jest nowsz wersj");
						retValue = FileComparison.File1IsNewer;
					}
					else if (file1Version.FileMinorPart < file2Version.FileMinorPart)
					{
                        Console.WriteLine(file2 + " jest nowsz wersj");
						retValue = FileComparison.File2IsNewer;
					}
                    else // poboczne numery wersji s rwne, sprawdzamy dalej...
					{
						// sprawdzenie numeru kompilacji
						if (file1Version.FileBuildPart > file2Version.FileBuildPart)
						{
                            Console.WriteLine(file1 + " jest nowsz wersj");
							retValue = FileComparison.File1IsNewer;
						}
						else if (file1Version.FileBuildPart < file2Version.FileBuildPart)
						{
                            Console.WriteLine(file2 + " jest nowsz wersj");
							retValue = FileComparison.File2IsNewer;
						}
                        else // numery kompilacji s rwne, sprawdzamy dalej...
						{
							// sprawdzenie numeru prywatnego
							if (file1Version.FilePrivatePart > file2Version.FilePrivatePart)
							{
                                Console.WriteLine(file1 + " jest nowsz wersj");
								retValue = FileComparison.File1IsNewer;
							}
							else if (file1Version.FilePrivatePart < 
								file2Version.FilePrivatePart)
							{
                                Console.WriteLine(file2 + " jest nowsz wersj");
								retValue = FileComparison.File2IsNewer;
							}
							else
							{
								// wersje identyczne
								Console.WriteLine("Wersja plikw jest identyczna");
								retValue = FileComparison.FilesAreSame;
							}
						}
					}
				}
			}
			return retValue;
		}

		#endregion

        #region 12.26 Uzyskiwanie informacji o wszystkich napdach obecnych w systemie
        public static void TestAllDriveInfo()
		{
			DisplayOneDrivesInfo();
			DisplayAllDriveInfo();
		}
		
		public static void DisplayOneDrivesInfo()
		{
			DriveInfo drive = new DriveInfo("D");
			if (drive.IsReady)
				Console.WriteLine("Ilo miejsca dostpnego w napdzie D:\\ : " + drive.AvailableFreeSpace);
			else
				Console.WriteLine("Napd D:\\ nie jest gotowy.");
	
			// Jeli napd nie bdzie gotowy, wyrzucony zostanie wyjtek
			//   System.IO.IOException: The device is not ready.	
			Console.WriteLine();
		}
				
		public static void DisplayAllDriveInfo()
		{
			foreach (DriveInfo drive in DriveInfo.GetDrives())
			{
				if (drive.IsReady)
				{
					Console.WriteLine("Napd " + drive.Name + " jest gotowy.");
					Console.WriteLine("Dostpne wolne miejsce: " + drive.AvailableFreeSpace);
					Console.WriteLine("Format napdu: " + drive.DriveFormat);
					Console.WriteLine("Typ napdu: " + drive.DriveType);
					Console.WriteLine("Nazwa: " + drive.Name);
					Console.WriteLine("Pena nazwa katalogu gwnego: " + drive.RootDirectory.FullName);
					Console.WriteLine("czna ilo wolnego miejsca: " + drive.TotalFreeSpace);
					Console.WriteLine("Cakowita pojemno: " + drive.TotalSize);
					Console.WriteLine("Etykieta napdu: " + drive.VolumeLabel);
				}
				else
				{
					Console.WriteLine("Napd " + drive.Name + " nie jest gotowy.");
				}
				
				Console.WriteLine();
			}
		}
		#endregion

        #region 12.27 Szyfrowanie i deszyfracja istniejcego pliku
        public static void TestEncryptExistingFile()
		{
			EncryptFile(@"C:\Encrypted.txt");
			DecryptFile(@"C:\Encrypted.txt");
		}

		public static void EncryptFile(string fileName)
		{
			File.Encrypt(fileName);
			// System.IO.IOException: The request is not supported.
			//at System.IO.__Error.WinIOError(Int32 errorCode, String str)
			//at System.IO.File.Encrypt(String path)
			//at CSharpRecipes.FileSystemIO.EncryptFile(String fileName) in c:\documents and settings\operator\desktop\csharp recipes 2nd edition\code\csharprecipes\12_filesystemio.cs:line 1987
			//at CSharpRecipes.FileSystemIO.TestEncryptExistingFile() in c:\documents and settings\operator\desktop\csharp recipes 2nd edition\code\csharprecipes\12_filesystemio.cs:line 1981
			//at CSharpRecipes.MainTester.TestFileSystemIO() in c:\documents and settings\operator\desktop\csharp recipes 2nd edition\code\csharprecipes\00_maintester.cs:line 786
			//at CSharpRecipes.MainTester.Main(String[] args) in c:\documents and settings\operator\desktop\csharp recipes 2nd edition\code\csharprecipes\00_maintester.cs:line 927

            // bool Microsoft.Win32.Win32Native::EncryptFile(string)
		}

		public static void DecryptFile(string fileName)
		{
			File.Decrypt(fileName);
		}
		#endregion

        #region 12.28 Kompresowanie i dekompresja plikw
        public static void TestCompressNewFile()
		{
			byte[] data = new byte[10000000];
			for (int i = 0; i < 10000000; i++)
				data[i] = (byte)i;
				
			using (FileStream fs = new FileStream(@"C:\NewNormalFile.txt", FileMode.OpenOrCreate, FileAccess.ReadWrite, FileShare.None))
			    NormalFile(fs, data);

			using (FileStream fs = new FileStream(@"C:\NewCompressedFile.txt", FileMode.OpenOrCreate, FileAccess.ReadWrite, FileShare.None))
			    CompressFile(fs, data, CompressionType.Deflate);

            using (FileStream fs = new FileStream(@"C:\NewCompressedFile.txt", FileMode.OpenOrCreate, FileAccess.ReadWrite, FileShare.None))
            {
                byte[] retData = DeCompressFile(fs, CompressionType.Deflate);
                Console.WriteLine("Rozmiar pliku skompresowanego Deflate == " + retData.Count);
            }
			
			using (FileStream fs = new FileStream(@"C:\NewGZCompressedFile.txt", FileMode.OpenOrCreate, FileAccess.ReadWrite, FileShare.None))
			    CompressFile(fs, data, CompressionType.GZip);

            using (FileStream fs = new FileStream(@"C:\NewGzCompressedFile.txt", FileMode.OpenOrCreate, FileAccess.ReadWrite, FileShare.None))
            {
                retData = DeCompressFile(fs, CompressionType.GZip);
                Console.WriteLine("Rozmiar pliku skompresowanego GZip == " + retData.Count);
            }
		}

        // metoda, ktra wypisuje rozmiar pliku przed skompresowaniem w celu porwnania wielkoci
		public static void NormalFile(Stream strm, byte[] data)
		{
			BinaryWriter normal = new BinaryWriter(strm);
			normal.Write(data);
			normal.Close();
		}

		public static void CompressFile(Stream strm, byte[] data, CompressionType compressionType)
		{
			// ustalenie sposobu kompresji pliku
			Stream deflate = null;
            if (compressionType == CompressionType.Deflate)
            {
                using (deflate = new DeflateStream(strm, CompressionMode.Compress))
                {
                    // zapisanie skompresowanych danych do pliku
                    deflate.Write(data, 0, data.Length);
                }
            }
            else
            {
                using (deflate = new GZipStream(strm, CompressionMode.Compress))
                {
                    // zapisanie skompresowanych danych do pliku
                    deflate.Write(data, 0, data.Length);
                }
            }
        }
     
        public static byte[] DeCompressFile(Stream strm, CompressionType compressionType)
		{
			// ustalenie sposobu dekompresji pliku
			Stream reInflate = null;

            if (compressionType == CompressionType.Deflate)
            {
                using (reInflate = new DeflateStream(strm, CompressionMode.Decompress))
                {
                    return (DeCompress(reInflate));
                }
            }
            else
            {
                using (reInflate = new GZipStream(strm, CompressionMode.Decompress))
                {
                    return (DeCompress(reInflate));
                }
            }
        }
			
		public static byte[] DeCompress(Stream reInflate)
        {
            List<byte> data = new List<byte>();
			int retVal = 0;

			// wczytanie wszystkich danych i ich dekompresja
			while (retVal >= 0)
			{
				retVal = reInflate.ReadByte();
				if (retVal != -1)
					data.Add((byte)retVal);
			}
				
			return (data.ToArray());
		}
		
		public enum CompressionType
		{
			Deflate,
			GZip
		}
		#endregion

	}
}
